use utf8;
use strict;

use XML::Pastor::Schema::Object;

package XML::Pastor::Schema::Element;
our @ISA = qw(XML::Pastor::Schema::Object);

XML::Pastor::Schema::Element->mk_accessors(qw(baseClasses minOccurs maxOccurs targetNamespace));


sub isSingleton {
	my $self = shift;
	
	my $maxOccurs = $self->maxOccurs();
	
	return 1 unless ($maxOccurs);
	return 0 if ($maxOccurs eq 'unbounded');
	return 1 if ($maxOccurs == 1);
	return 0;			
}

1;

__END__

=head1 NAME

B<XML::Pastor::Schema::Group> - Class that represents the META information about a W3C schema B<group>.

=head1 WARNING

This module is used internally by L<XML::Pastor>. You do not normally know much about this module to actually use L<XML::Pastor>.  It is 
documented here for completeness and for L<XML::Pastor> developers. Do not count on the interface of this module. It may change in 
any of the subsequent releases. You have been warned. 

=head1 ISA

This class descends from L<XML::Pastor::Schema::Object>.


=head1 SYNOPSIS
  
  my $g = XML::Pastor::Schema::Group->new();
  
  $g->setFields(name => 'personal', scope=> 'global', nameIsAutoGenerated=>0);

  $g->elements(['lastName', 'firstName', 'title', 'dateOfBirth']);
  
  print $g->name();	# prints 'personal'.
  print $g->scope();# prints 'global'.
  

=head1 DESCRIPTION

B<XML::Pastor::Schema::Group> is a data-oriented object class that reprsents a W3C B<group>. It is
parsed from the W3C schema and is used a building block for the produced B<schema model>. Objects of this 
class contain META information about the W3C schema B<group> that they represent. 

Like other schema object classes, this is a data-oriented object class, meaning it doesn't have many methods other 
than a constructor and various accessors. 

=head1 METHODS

=head2 CONSTRUCTORS
 
=head4 new() 

  $class->new(%fields)

B<CONSTRUCTOR>, overriden. 

The new() constructor method instantiates a new object. It is inheritable. 
  
Any -named- fields that are passed as parameters are initialized to those values within
the newly created object. 

In its overriden form, what this method does is as follows:

=over

=item * sets the I<contentType> field to 'I<group>';

=item * creates the B<elements> array-ref field if not passed already as a parameter;

=item * creates the B<elementInfo> hash-ref field if not passed already as a parameter;

=back

.

=head2 ACCESSORS

=head3 Inherited accessors

Several accessors are inherited by this class from its ancestor L<XML::Pastor::Schema::Object>. 
Please see L<XML::Pastor::Schema::Object> for a documentation of those.

=head3 Accessors defined here

=head4 baseClasses()

  my $bases = $object->baseClasses();	# GET
  $object->baseClasses($bases);  	    # SET

The base classes of this element class, when it is generated by L<XML::Pastor>. This value is
computed at schema model I<resolution> time by L<XML::Pastor::Schema::Model/resolve()>.

=head4 maxOccurs()

  my $maxo = $object->maxOccurs();	# GET
  $object->maxOccurs($maxo);  	    # SET

This is a W3C property.

The maximum allowed occurences of this child element. This can be any non-negative integer or the string
'I<unbounded>'. 

=head4 minOccurs()

  my $mino = $object->minOccurs();	# GET
  $object->minOccurs($mino);  	    # SET

This is a W3C property.

The minimum allowed occurences of this child element. This can be any non-negative integer. 
When it is zero (0), this means the occurence of this child element is optional.

.

=head2 OTHER METHODS

=head4 isSingleton()

This returns a boolean value. When TRUE(1), this means that the child element is a singleton, meaning
the B<maxOccurs> property is either undefined or it is exactly 1. 

.

=head1 BUGS & CAVEATS

There no known bugs at this time, but this doesn't mean there are aren't any. 
Note that, although some testing was done prior to releasing the module, this should still be considered alpha code. 
So use it at your own risk.

Note that there may be other bugs or limitations that the author is not aware of.

=head1 AUTHOR

Ayhan Ulusoy <dev@ulusoy.name>


=head1 COPYRIGHT

  Copyright (C) 2006-2007 Ayhan Ulusoy. All Rights Reserved.

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.


=head1 SEE ALSO

See also L<XML::Pastor>, L<XML::Pastor::ComplexType>, L<XML::Pastor::SimpleType>

If you are curious about the implementation, see L<XML::Pastor::Schema::Parser>,
L<XML::Pastor::Schema::Model>, L<XML::Pastor::Generator>.

If you really want to dig in, see L<XML::Pastor::Schema::Attribute>, L<XML::Pastor::Schema::AttributeGroup>,
L<XML::Pastor::Schema::ComplexType>, L<XML::Pastor::Schema::Element>, L<XML::Pastor::Schema::Group>,
L<XML::Pastor::Schema::List>, L<XML::Pastor::Schema::SimpleType>, L<XML::Pastor::Schema::Type>, 
L<XML::Pastor::Schema::Object>

=cut

