#!/usr/bin/env perl

use strict;

use Test::More tests => 4386;

use FindBin;
use lib "$FindBin::Bin/lib";
use My_Growth_Test;

require_ok('Medical::Growth::NHANES_2000::BMI_for_Age::Child::Female');

my $h = Medical::Growth::NHANES_2000::BMI_for_Age::Child::Female->new;
My_Growth_Test::run_data_tests($h);

is( $h->bmi( 20, 100 ), 20, 'BMI computation: 20' );
is( $h->bmi( 30, 100 ), 30, 'BMI computation: 30' );
is( sprintf( '%5.2f', $h->bmi( 20, 110 ) ), '16.53', 'BMI computation: 16.53' );
ok( !defined( $h->bmi( undef, 110 ) ), 'BMI computation: missing weight' );
ok( !defined( $h->bmi(20) ), 'BMI computation: missing height' );

__DATA__
# age %ile value
 24.0	 3	 14.147
 24.0	 5	 14.398
 24.0	10	 14.801
 24.0	25	 15.528
 24.0	50	 16.423
 24.0	75	 17.427
 24.0	85	 18.018
 24.0	90	 18.441
 24.0	95	 19.106
 24.0	97	 19.564
 24.5	 3	 14.132
 24.5	 5	 14.380
 24.5	10	 14.780
 24.5	25	 15.500
 24.5	50	 16.388
 24.5	75	 17.386
 24.5	85	 17.974
 24.5	90	 18.395
 24.5	95	 19.058
 24.5	97	 19.515
 25.5	 3	 14.102
 25.5	 5	 14.345
 25.5	10	 14.737
 25.5	25	 15.444
 25.5	50	 16.319
 25.5	75	 17.305
 25.5	85	 17.887
 25.5	90	 18.306
 25.5	95	 18.966
 25.5	97	 19.422
 26.5	 3	 14.073
 26.5	 5	 14.311
 26.5	10	 14.695
 26.5	25	 15.390
 26.5	50	 16.252
 26.5	75	 17.227
 26.5	85	 17.805
 26.5	90	 18.221
 26.5	95	 18.879
 26.5	97	 19.334
 27.5	 3	 14.044
 27.5	 5	 14.277
 27.5	10	 14.654
 27.5	25	 15.338
 27.5	50	 16.187
 27.5	75	 17.152
 27.5	85	 17.726
 27.5	90	 18.140
 27.5	95	 18.796
 27.5	97	 19.252
 28.5	 3	 14.015
 28.5	 5	 14.244
 28.5	10	 14.614
 28.5	25	 15.286
 28.5	50	 16.125
 28.5	75	 17.080
 28.5	85	 17.650
 28.5	90	 18.063
 28.5	95	 18.718
 28.5	97	 19.174
 29.5	 3	 13.987
 29.5	 5	 14.212
 29.5	10	 14.575
 29.5	25	 15.237
 29.5	50	 16.064
 29.5	75	 17.011
 29.5	85	 17.578
 29.5	90	 17.990
 29.5	95	 18.645
 29.5	97	 19.103
 30.5	 3	 13.960
 30.5	 5	 14.180
 30.5	10	 14.537
 30.5	25	 15.188
 30.5	50	 16.006
 30.5	75	 16.945
 30.5	85	 17.510
 30.5	90	 17.920
 30.5	95	 18.576
 30.5	97	 19.036
 31.5	 3	 13.932
 31.5	 5	 14.149
 31.5	10	 14.500
 31.5	25	 15.142
 31.5	50	 15.950
 31.5	75	 16.882
 31.5	85	 17.444
 31.5	90	 17.855
 31.5	95	 18.512
 31.5	97	 18.974
 32.5	 3	 13.905
 32.5	 5	 14.118
 32.5	10	 14.464
 32.5	25	 15.096
 32.5	50	 15.895
 32.5	75	 16.821
 32.5	85	 17.382
 32.5	90	 17.793
 32.5	95	 18.452
 32.5	97	 18.917
 33.5	 3	 13.879
 33.5	 5	 14.088
 33.5	10	 14.428
 33.5	25	 15.053
 33.5	50	 15.843
 33.5	75	 16.764
 33.5	85	 17.324
 33.5	90	 17.734
 33.5	95	 18.396
 33.5	97	 18.865
 34.5	 3	 13.853
 34.5	 5	 14.059
 34.5	10	 14.394
 34.5	25	 15.010
 34.5	50	 15.793
 34.5	75	 16.709
 34.5	85	 17.268
 34.5	90	 17.679
 34.5	95	 18.345
 34.5	97	 18.818
 35.5	 3	 13.827
 35.5	 5	 14.030
 35.5	10	 14.361
 35.5	25	 14.969
 35.5	50	 15.745
 35.5	75	 16.656
 35.5	85	 17.216
 35.5	90	 17.628
 35.5	95	 18.298
 35.5	97	 18.776
 36.5	 3	 13.802
 36.5	 5	 14.002
 36.5	10	 14.328
 36.5	25	 14.930
 36.5	50	 15.699
 36.5	75	 16.607
 36.5	85	 17.166
 36.5	90	 17.580
 36.5	95	 18.255
 36.5	97	 18.738
 37.5	 3	 13.778
 37.5	 5	 13.975
 37.5	10	 14.296
 37.5	25	 14.891
 37.5	50	 15.655
 37.5	75	 16.560
 37.5	85	 17.120
 37.5	90	 17.536
 37.5	95	 18.216
 37.5	97	 18.705
 38.5	 3	 13.753
 38.5	 5	 13.948
 38.5	10	 14.266
 38.5	25	 14.855
 38.5	50	 15.613
 38.5	75	 16.516
 38.5	85	 17.077
 38.5	90	 17.495
 38.5	95	 18.181
 38.5	97	 18.676
 39.5	 3	 13.730
 39.5	 5	 13.922
 39.5	10	 14.236
 39.5	25	 14.819
 39.5	50	 15.573
 39.5	75	 16.474
 39.5	85	 17.037
 39.5	90	 17.457
 39.5	95	 18.149
 39.5	97	 18.652
 40.5	 3	 13.706
 40.5	 5	 13.896
 40.5	10	 14.207
 40.5	25	 14.785
 40.5	50	 15.535
 40.5	75	 16.435
 40.5	85	 16.999
 40.5	90	 17.422
 40.5	95	 18.122
 40.5	97	 18.632
 41.5	 3	 13.683
 41.5	 5	 13.871
 41.5	10	 14.179
 41.5	25	 14.753
 41.5	50	 15.499
 41.5	75	 16.398
 41.5	85	 16.965
 41.5	90	 17.390
 41.5	95	 18.098
 41.5	97	 18.616
 42.5	 3	 13.661
 42.5	 5	 13.847
 42.5	10	 14.152
 42.5	25	 14.722
 42.5	50	 15.465
 42.5	75	 16.364
 42.5	85	 16.933
 42.5	90	 17.362
 42.5	95	 18.078
 42.5	97	 18.604
 43.5	 3	 13.639
 43.5	 5	 13.823
 43.5	10	 14.126
 43.5	25	 14.692
 43.5	50	 15.432
 43.5	75	 16.332
 43.5	85	 16.904
 43.5	90	 17.337
 43.5	95	 18.061
 43.5	97	 18.596
 44.5	 3	 13.618
 44.5	 5	 13.800
 44.5	10	 14.101
 44.5	25	 14.664
 44.5	50	 15.402
 44.5	75	 16.303
 44.5	85	 16.878
 44.5	90	 17.314
 44.5	95	 18.048
 44.5	97	 18.593
 45.5	 3	 13.597
 45.5	 5	 13.778
 45.5	10	 14.077
 45.5	25	 14.637
 45.5	50	 15.373
 45.5	75	 16.276
 45.5	85	 16.855
 45.5	90	 17.295
 45.5	95	 18.038
 45.5	97	 18.592
 46.5	 3	 13.576
 46.5	 5	 13.756
 46.5	10	 14.053
 46.5	25	 14.611
 46.5	50	 15.347
 46.5	75	 16.252
 46.5	85	 16.834
 46.5	90	 17.279
 46.5	95	 18.032
 46.5	97	 18.596
 47.5	 3	 13.556
 47.5	 5	 13.735
 47.5	10	 14.031
 47.5	25	 14.587
 47.5	50	 15.322
 47.5	75	 16.230
 47.5	85	 16.816
 47.5	90	 17.265
 47.5	95	 18.028
 47.5	97	 18.603
 48.5	 3	 13.536
 48.5	 5	 13.715
 48.5	10	 14.009
 48.5	25	 14.563
 48.5	50	 15.299
 48.5	75	 16.210
 48.5	85	 16.801
 48.5	90	 17.254
 48.5	95	 18.029
 48.5	97	 18.614
 49.5	 3	 13.517
 49.5	 5	 13.695
 49.5	10	 13.988
 49.5	25	 14.542
 49.5	50	 15.277
 49.5	75	 16.192
 49.5	85	 16.788
 49.5	90	 17.246
 49.5	95	 18.032
 49.5	97	 18.629
 50.5	 3	 13.499
 50.5	 5	 13.676
 50.5	10	 13.968
 50.5	25	 14.521
 50.5	50	 15.258
 50.5	75	 16.177
 50.5	85	 16.777
 50.5	90	 17.241
 50.5	95	 18.038
 50.5	97	 18.647
 51.5	 3	 13.481
 51.5	 5	 13.657
 51.5	10	 13.949
 51.5	25	 14.502
 51.5	50	 15.240
 51.5	75	 16.164
 51.5	85	 16.769
 51.5	90	 17.238
 51.5	95	 18.047
 51.5	97	 18.668
 52.5	 3	 13.463
 52.5	 5	 13.639
 52.5	10	 13.931
 52.5	25	 14.484
 52.5	50	 15.223
 52.5	75	 16.153
 52.5	85	 16.764
 52.5	90	 17.238
 52.5	95	 18.060
 52.5	97	 18.692
 53.5	 3	 13.446
 53.5	 5	 13.622
 53.5	10	 13.914
 53.5	25	 14.467
 53.5	50	 15.209
 53.5	75	 16.144
 53.5	85	 16.761
 53.5	90	 17.241
 53.5	95	 18.075
 53.5	97	 18.720
 54.5	 3	 13.430
 54.5	 5	 13.606
 54.5	10	 13.898
 54.5	25	 14.452
 54.5	50	 15.196
 54.5	75	 16.137
 54.5	85	 16.760
 54.5	90	 17.246
 54.5	95	 18.093
 54.5	97	 18.751
 55.5	 3	 13.414
 55.5	 5	 13.590
 55.5	10	 13.882
 55.5	25	 14.437
 55.5	50	 15.185
 55.5	75	 16.132
 55.5	85	 16.761
 55.5	90	 17.253
 55.5	95	 18.114
 55.5	97	 18.784
 56.5	 3	 13.398
 56.5	 5	 13.575
 56.5	10	 13.867
 56.5	25	 14.424
 56.5	50	 15.175
 56.5	75	 16.130
 56.5	85	 16.765
 56.5	90	 17.263
 56.5	95	 18.137
 56.5	97	 18.821
 57.5	 3	 13.384
 57.5	 5	 13.560
 57.5	10	 13.854
 57.5	25	 14.412
 57.5	50	 15.167
 57.5	75	 16.129
 57.5	85	 16.771
 57.5	90	 17.276
 57.5	95	 18.163
 57.5	97	 18.861
 58.5	 3	 13.369
 58.5	 5	 13.546
 58.5	10	 13.841
 58.5	25	 14.402
 58.5	50	 15.160
 58.5	75	 16.130
 58.5	85	 16.779
 58.5	90	 17.290
 58.5	95	 18.192
 58.5	97	 18.903
 59.5	 3	 13.356
 59.5	 5	 13.533
 59.5	10	 13.829
 59.5	25	 14.392
 59.5	50	 15.155
 59.5	75	 16.133
 59.5	85	 16.790
 59.5	90	 17.307
 59.5	95	 18.224
 59.5	97	 18.948
 60.5	 3	 13.343
 60.5	 5	 13.521
 60.5	10	 13.817
 60.5	25	 14.383
 60.5	50	 15.152
 60.5	75	 16.138
 60.5	85	 16.802
 60.5	90	 17.327
 60.5	95	 18.257
 60.5	97	 18.996
 61.5	 3	 13.330
 61.5	 5	 13.509
 61.5	10	 13.807
 61.5	25	 14.376
 61.5	50	 15.150
 61.5	75	 16.145
 61.5	85	 16.816
 61.5	90	 17.348
 61.5	95	 18.294
 61.5	97	 19.047
 62.5	 3	 13.318
 62.5	 5	 13.498
 62.5	10	 13.797
 62.5	25	 14.370
 62.5	50	 15.149
 62.5	75	 16.154
 62.5	85	 16.833
 62.5	90	 17.372
 62.5	95	 18.332
 62.5	97	 19.100
 63.5	 3	 13.307
 63.5	 5	 13.488
 63.5	10	 13.789
 63.5	25	 14.365
 63.5	50	 15.150
 63.5	75	 16.164
 63.5	85	 16.851
 63.5	90	 17.397
 63.5	95	 18.373
 63.5	97	 19.155
 64.5	 3	 13.296
 64.5	 5	 13.478
 64.5	10	 13.781
 64.5	25	 14.361
 64.5	50	 15.152
 64.5	75	 16.177
 64.5	85	 16.872
 64.5	90	 17.425
 64.5	95	 18.417
 64.5	97	 19.213
 65.5	 3	 13.286
 65.5	 5	 13.469
 65.5	10	 13.774
 65.5	25	 14.358
 65.5	50	 15.156
 65.5	75	 16.191
 65.5	85	 16.894
 65.5	90	 17.455
 65.5	95	 18.462
 65.5	97	 19.273
 66.5	 3	 13.277
 66.5	 5	 13.461
 66.5	10	 13.767
 66.5	25	 14.356
 66.5	50	 15.161
 66.5	75	 16.206
 66.5	85	 16.918
 66.5	90	 17.487
 66.5	95	 18.510
 66.5	97	 19.336
 67.5	 3	 13.268
 67.5	 5	 13.453
 67.5	10	 13.762
 67.5	25	 14.355
 67.5	50	 15.167
 67.5	75	 16.223
 67.5	85	 16.944
 67.5	90	 17.521
 67.5	95	 18.559
 67.5	97	 19.400
 68.5	 3	 13.260
 68.5	 5	 13.446
 68.5	10	 13.757
 68.5	25	 14.355
 68.5	50	 15.174
 68.5	75	 16.242
 68.5	85	 16.972
 68.5	90	 17.556
 68.5	95	 18.611
 68.5	97	 19.467
 69.5	 3	 13.252
 69.5	 5	 13.440
 69.5	10	 13.753
 69.5	25	 14.356
 69.5	50	 15.183
 69.5	75	 16.263
 69.5	85	 17.001
 69.5	90	 17.594
 69.5	95	 18.665
 69.5	97	 19.536
 70.5	 3	 13.245
 70.5	 5	 13.435
 70.5	10	 13.750
 70.5	25	 14.358
 70.5	50	 15.193
 70.5	75	 16.284
 70.5	85	 17.032
 70.5	90	 17.633
 70.5	95	 18.721
 70.5	97	 19.607
 71.5	 3	 13.239
 71.5	 5	 13.430
 71.5	10	 13.748
 71.5	25	 14.361
 71.5	50	 15.204
 71.5	75	 16.308
 71.5	85	 17.065
 71.5	90	 17.674
 71.5	95	 18.778
 71.5	97	 19.680
 72.5	 3	 13.233
 72.5	 5	 13.426
 72.5	10	 13.747
 72.5	25	 14.366
 72.5	50	 15.217
 72.5	75	 16.333
 72.5	85	 17.100
 72.5	90	 17.717
 72.5	95	 18.838
 72.5	97	 19.755
 73.5	 3	 13.229
 73.5	 5	 13.423
 73.5	10	 13.746
 73.5	25	 14.371
 73.5	50	 15.231
 73.5	75	 16.359
 73.5	85	 17.136
 73.5	90	 17.761
 73.5	95	 18.899
 73.5	97	 19.831
 74.5	 3	 13.224
 74.5	 5	 13.420
 74.5	10	 13.747
 74.5	25	 14.377
 74.5	50	 15.245
 74.5	75	 16.387
 74.5	85	 17.173
 74.5	90	 17.807
 74.5	95	 18.962
 74.5	97	 19.910
 75.5	 3	 13.221
 75.5	 5	 13.418
 75.5	10	 13.748
 75.5	25	 14.384
 75.5	50	 15.261
 75.5	75	 16.416
 75.5	85	 17.212
 75.5	90	 17.855
 75.5	95	 19.027
 75.5	97	 19.990
 76.5	 3	 13.218
 76.5	 5	 13.417
 76.5	10	 13.749
 76.5	25	 14.392
 76.5	50	 15.279
 76.5	75	 16.446
 76.5	85	 17.253
 76.5	90	 17.904
 76.5	95	 19.093
 76.5	97	 20.072
 77.5	 3	 13.215
 77.5	 5	 13.416
 77.5	10	 13.752
 77.5	25	 14.401
 77.5	50	 15.297
 77.5	75	 16.478
 77.5	85	 17.295
 77.5	90	 17.955
 77.5	95	 19.161
 77.5	97	 20.155
 78.5	 3	 13.214
 78.5	 5	 13.417
 78.5	10	 13.755
 78.5	25	 14.410
 78.5	50	 15.316
 78.5	75	 16.511
 78.5	85	 17.338
 78.5	90	 18.007
 78.5	95	 19.231
 78.5	97	 20.240
 79.5	 3	 13.213
 79.5	 5	 13.417
 79.5	10	 13.760
 79.5	25	 14.421
 79.5	50	 15.336
 79.5	75	 16.545
 79.5	85	 17.383
 79.5	90	 18.061
 79.5	95	 19.302
 79.5	97	 20.327
 80.5	 3	 13.213
 80.5	 5	 13.419
 80.5	10	 13.765
 80.5	25	 14.433
 80.5	50	 15.358
 80.5	75	 16.581
 80.5	85	 17.429
 80.5	90	 18.116
 80.5	95	 19.374
 80.5	97	 20.415
 81.5	 3	 13.213
 81.5	 5	 13.421
 81.5	10	 13.770
 81.5	25	 14.445
 81.5	50	 15.380
 81.5	75	 16.618
 81.5	85	 17.476
 81.5	90	 18.172
 81.5	95	 19.448
 81.5	97	 20.504
 82.5	 3	 13.214
 82.5	 5	 13.425
 82.5	10	 13.777
 82.5	25	 14.458
 82.5	50	 15.404
 82.5	75	 16.656
 82.5	85	 17.525
 82.5	90	 18.230
 82.5	95	 19.523
 82.5	97	 20.595
 83.5	 3	 13.216
 83.5	 5	 13.428
 83.5	10	 13.784
 83.5	25	 14.473
 83.5	50	 15.428
 83.5	75	 16.695
 83.5	85	 17.575
 83.5	90	 18.289
 83.5	95	 19.600
 83.5	97	 20.687
 84.5	 3	 13.218
 84.5	 5	 13.433
 84.5	10	 13.792
 84.5	25	 14.488
 84.5	50	 15.454
 84.5	75	 16.735
 84.5	85	 17.626
 84.5	90	 18.349
 84.5	95	 19.678
 84.5	97	 20.781
 85.5	 3	 13.221
 85.5	 5	 13.438
 85.5	10	 13.801
 85.5	25	 14.504
 85.5	50	 15.480
 85.5	75	 16.776
 85.5	85	 17.678
 85.5	90	 18.410
 85.5	95	 19.757
 85.5	97	 20.875
 86.5	 3	 13.225
 86.5	 5	 13.444
 86.5	10	 13.810
 86.5	25	 14.520
 86.5	50	 15.507
 86.5	75	 16.818
 86.5	85	 17.731
 86.5	90	 18.473
 86.5	95	 19.837
 86.5	97	 20.971
 87.5	 3	 13.229
 87.5	 5	 13.450
 87.5	10	 13.820
 87.5	25	 14.538
 87.5	50	 15.535
 87.5	75	 16.861
 87.5	85	 17.785
 87.5	90	 18.536
 87.5	95	 19.919
 87.5	97	 21.068
 88.5	 3	 13.235
 88.5	 5	 13.458
 88.5	10	 13.831
 88.5	25	 14.556
 88.5	50	 15.564
 88.5	75	 16.906
 88.5	85	 17.841
 88.5	90	 18.601
 88.5	95	 20.001
 88.5	97	 21.166
 89.5	 3	 13.240
 89.5	 5	 13.466
 89.5	10	 13.843
 89.5	25	 14.575
 89.5	50	 15.594
 89.5	75	 16.951
 89.5	85	 17.897
 89.5	90	 18.667
 89.5	95	 20.085
 89.5	97	 21.265
 90.5	 3	 13.247
 90.5	 5	 13.474
 90.5	10	 13.855
 90.5	25	 14.595
 90.5	50	 15.625
 90.5	75	 16.997
 90.5	85	 17.954
 90.5	90	 18.733
 90.5	95	 20.169
 90.5	97	 21.364
 91.5	 3	 13.254
 91.5	 5	 13.484
 91.5	10	 13.868
 91.5	25	 14.616
 91.5	50	 15.657
 91.5	75	 17.044
 91.5	85	 18.013
 91.5	90	 18.801
 91.5	95	 20.255
 91.5	97	 21.465
 92.5	 3	 13.262
 92.5	 5	 13.494
 92.5	10	 13.882
 92.5	25	 14.637
 92.5	50	 15.689
 92.5	75	 17.092
 92.5	85	 18.072
 92.5	90	 18.870
 92.5	95	 20.341
 92.5	97	 21.567
 93.5	 3	 13.270
 93.5	 5	 13.504
 93.5	10	 13.897
 93.5	25	 14.659
 93.5	50	 15.723
 93.5	75	 17.141
 93.5	85	 18.132
 93.5	90	 18.939
 93.5	95	 20.428
 93.5	97	 21.669
 94.5	 3	 13.279
 94.5	 5	 13.516
 94.5	10	 13.912
 94.5	25	 14.682
 94.5	50	 15.757
 94.5	75	 17.190
 94.5	85	 18.193
 94.5	90	 19.009
 94.5	95	 20.517
 94.5	97	 21.773
 95.5	 3	 13.289
 95.5	 5	 13.528
 95.5	10	 13.928
 95.5	25	 14.706
 95.5	50	 15.791
 95.5	75	 17.241
 95.5	85	 18.255
 95.5	90	 19.081
 95.5	95	 20.606
 95.5	97	 21.877
 96.5	 3	 13.300
 96.5	 5	 13.541
 96.5	10	 13.944
 96.5	25	 14.730
 96.5	50	 15.827
 96.5	75	 17.292
 96.5	85	 18.317
 96.5	90	 19.153
 96.5	95	 20.695
 96.5	97	 21.981
 97.5	 3	 13.311
 97.5	 5	 13.554
 97.5	10	 13.962
 97.5	25	 14.755
 97.5	50	 15.863
 97.5	75	 17.344
 97.5	85	 18.381
 97.5	90	 19.225
 97.5	95	 20.786
 97.5	97	 22.087
 98.5	 3	 13.323
 98.5	 5	 13.568
 98.5	10	 13.980
 98.5	25	 14.781
 98.5	50	 15.900
 98.5	75	 17.397
 98.5	85	 18.445
 98.5	90	 19.299
 98.5	95	 20.877
 98.5	97	 22.193
 99.5	 3	 13.335
 99.5	 5	 13.583
 99.5	10	 13.998
 99.5	25	 14.807
 99.5	50	 15.938
 99.5	75	 17.450
 99.5	85	 18.509
 99.5	90	 19.373
 99.5	95	 20.969
 99.5	97	 22.299
100.5	 3	 13.348
100.5	 5	 13.598
100.5	10	 14.018
100.5	25	 14.834
100.5	50	 15.976
100.5	75	 17.504
100.5	85	 18.575
100.5	90	 19.448
100.5	95	 21.061
100.5	97	 22.406
101.5	 3	 13.362
101.5	 5	 13.614
101.5	10	 14.038
101.5	25	 14.862
101.5	50	 16.015
101.5	75	 17.559
101.5	85	 18.641
101.5	90	 19.523
101.5	95	 21.154
101.5	97	 22.514
102.5	 3	 13.376
102.5	 5	 13.631
102.5	10	 14.058
102.5	25	 14.890
102.5	50	 16.055
102.5	75	 17.615
102.5	85	 18.708
102.5	90	 19.599
102.5	95	 21.247
102.5	97	 22.622
103.5	 3	 13.391
103.5	 5	 13.648
103.5	10	 14.079
103.5	25	 14.919
103.5	50	 16.096
103.5	75	 17.671
103.5	85	 18.775
103.5	90	 19.676
103.5	95	 21.341
103.5	97	 22.730
104.5	 3	 13.407
104.5	 5	 13.666
104.5	10	 14.101
104.5	25	 14.949
104.5	50	 16.136
104.5	75	 17.727
104.5	85	 18.843
104.5	90	 19.753
104.5	95	 21.436
104.5	97	 22.839
105.5	 3	 13.423
105.5	 5	 13.685
105.5	10	 14.124
105.5	25	 14.979
105.5	50	 16.178
105.5	75	 17.784
105.5	85	 18.911
105.5	90	 19.831
105.5	95	 21.530
105.5	97	 22.948
106.5	 3	 13.440
106.5	 5	 13.704
106.5	10	 14.147
106.5	25	 15.010
106.5	50	 16.220
106.5	75	 17.842
106.5	85	 18.980
106.5	90	 19.909
106.5	95	 21.626
106.5	97	 23.057
107.5	 3	 13.458
107.5	 5	 13.724
107.5	10	 14.170
107.5	25	 15.042
107.5	50	 16.263
107.5	75	 17.900
107.5	85	 19.050
107.5	90	 19.987
107.5	95	 21.721
107.5	97	 23.167
108.5	 3	 13.476
108.5	 5	 13.744
108.5	10	 14.195
108.5	25	 15.074
108.5	50	 16.306
108.5	75	 17.959
108.5	85	 19.119
108.5	90	 20.066
108.5	95	 21.817
108.5	97	 23.277
109.5	 3	 13.495
109.5	 5	 13.765
109.5	10	 14.220
109.5	25	 15.106
109.5	50	 16.350
109.5	75	 18.018
109.5	85	 19.190
109.5	90	 20.146
109.5	95	 21.913
109.5	97	 23.387
110.5	 3	 13.514
110.5	 5	 13.787
110.5	10	 14.245
110.5	25	 15.140
110.5	50	 16.394
110.5	75	 18.078
110.5	85	 19.260
110.5	90	 20.226
110.5	95	 22.010
110.5	97	 23.498
111.5	 3	 13.534
111.5	 5	 13.809
111.5	10	 14.271
111.5	25	 15.173
111.5	50	 16.439
111.5	75	 18.138
111.5	85	 19.331
111.5	90	 20.306
111.5	95	 22.107
111.5	97	 23.608
112.5	 3	 13.555
112.5	 5	 13.832
112.5	10	 14.298
112.5	25	 15.208
112.5	50	 16.484
112.5	75	 18.199
112.5	85	 19.403
112.5	90	 20.386
112.5	95	 22.204
112.5	97	 23.719
113.5	 3	 13.576
113.5	 5	 13.855
113.5	10	 14.325
113.5	25	 15.242
113.5	50	 16.530
113.5	75	 18.260
113.5	85	 19.475
113.5	90	 20.467
113.5	95	 22.301
113.5	97	 23.829
114.5	 3	 13.598
114.5	 5	 13.879
114.5	10	 14.353
114.5	25	 15.278
114.5	50	 16.576
114.5	75	 18.321
114.5	85	 19.547
114.5	90	 20.548
114.5	95	 22.398
114.5	97	 23.940
115.5	 3	 13.620
115.5	 5	 13.904
115.5	10	 14.381
115.5	25	 15.314
115.5	50	 16.623
115.5	75	 18.383
115.5	85	 19.619
115.5	90	 20.629
115.5	95	 22.495
115.5	97	 24.051
116.5	 3	 13.643
116.5	 5	 13.929
116.5	10	 14.410
116.5	25	 15.350
116.5	50	 16.670
116.5	75	 18.444
116.5	85	 19.692
116.5	90	 20.710
116.5	95	 22.593
116.5	97	 24.161
117.5	 3	 13.667
117.5	 5	 13.955
117.5	10	 14.440
117.5	25	 15.387
117.5	50	 16.718
117.5	75	 18.507
117.5	85	 19.764
117.5	90	 20.791
117.5	95	 22.690
117.5	97	 24.272
118.5	 3	 13.691
118.5	 5	 13.981
118.5	10	 14.470
118.5	25	 15.424
118.5	50	 16.765
118.5	75	 18.569
118.5	85	 19.837
118.5	90	 20.873
118.5	95	 22.788
118.5	97	 24.383
119.5	 3	 13.716
119.5	 5	 14.008
119.5	10	 14.500
119.5	25	 15.462
119.5	50	 16.814
119.5	75	 18.632
119.5	85	 19.911
119.5	90	 20.955
119.5	95	 22.885
119.5	97	 24.493
120.5	 3	 13.741
120.5	 5	 14.035
120.5	10	 14.531
120.5	25	 15.500
120.5	50	 16.862
120.5	75	 18.695
120.5	85	 19.984
120.5	90	 21.037
120.5	95	 22.983
120.5	97	 24.603
121.5	 3	 13.767
121.5	 5	 14.063
121.5	10	 14.562
121.5	25	 15.539
121.5	50	 16.911
121.5	75	 18.759
121.5	85	 20.058
121.5	90	 21.119
121.5	95	 23.080
121.5	97	 24.714
122.5	 3	 13.793
122.5	 5	 14.092
122.5	10	 14.594
122.5	25	 15.578
122.5	50	 16.961
122.5	75	 18.822
122.5	85	 20.131
122.5	90	 21.201
122.5	95	 23.177
122.5	97	 24.824
123.5	 3	 13.820
123.5	 5	 14.121
123.5	10	 14.627
123.5	25	 15.617
123.5	50	 17.010
123.5	75	 18.886
123.5	85	 20.205
123.5	90	 21.283
123.5	95	 23.275
123.5	97	 24.933
124.5	 3	 13.847
124.5	 5	 14.150
124.5	10	 14.660
124.5	25	 15.657
124.5	50	 17.060
124.5	75	 18.950
124.5	85	 20.279
124.5	90	 21.365
124.5	95	 23.372
124.5	97	 25.043
125.5	 3	 13.875
125.5	 5	 14.180
125.5	10	 14.693
125.5	25	 15.697
125.5	50	 17.110
125.5	75	 19.014
125.5	85	 20.353
125.5	90	 21.447
125.5	95	 23.469
125.5	97	 25.152
126.5	 3	 13.903
126.5	 5	 14.210
126.5	10	 14.727
126.5	25	 15.738
126.5	50	 17.161
126.5	75	 19.078
126.5	85	 20.427
126.5	90	 21.529
126.5	95	 23.565
126.5	97	 25.261
127.5	 3	 13.932
127.5	 5	 14.241
127.5	10	 14.761
127.5	25	 15.779
127.5	50	 17.212
127.5	75	 19.142
127.5	85	 20.501
127.5	90	 21.610
127.5	95	 23.662
127.5	97	 25.370
128.5	 3	 13.962
128.5	 5	 14.273
128.5	10	 14.796
128.5	25	 15.820
128.5	50	 17.263
128.5	75	 19.206
128.5	85	 20.574
128.5	90	 21.692
128.5	95	 23.758
128.5	97	 25.478
129.5	 3	 13.992
129.5	 5	 14.305
129.5	10	 14.831
129.5	25	 15.862
129.5	50	 17.314
129.5	75	 19.271
129.5	85	 20.648
129.5	90	 21.774
129.5	95	 23.855
129.5	97	 25.587
130.5	 3	 14.022
130.5	 5	 14.337
130.5	10	 14.867
130.5	25	 15.904
130.5	50	 17.366
130.5	75	 19.335
130.5	85	 20.722
130.5	90	 21.856
130.5	95	 23.950
130.5	97	 25.694
131.5	 3	 14.053
131.5	 5	 14.370
131.5	10	 14.903
131.5	25	 15.946
131.5	50	 17.417
131.5	75	 19.400
131.5	85	 20.796
131.5	90	 21.937
131.5	95	 24.046
131.5	97	 25.802
132.5	 3	 14.085
132.5	 5	 14.403
132.5	10	 14.939
132.5	25	 15.989
132.5	50	 17.469
132.5	75	 19.465
132.5	85	 20.870
132.5	90	 22.018
132.5	95	 24.141
132.5	97	 25.909
133.5	 3	 14.116
133.5	 5	 14.437
133.5	10	 14.976
133.5	25	 16.032
133.5	50	 17.521
133.5	75	 19.529
133.5	85	 20.943
133.5	90	 22.099
133.5	95	 24.236
133.5	97	 26.015
134.5	 3	 14.149
134.5	 5	 14.471
134.5	10	 15.013
134.5	25	 16.075
134.5	50	 17.573
134.5	75	 19.594
134.5	85	 21.017
134.5	90	 22.180
134.5	95	 24.331
134.5	97	 26.121
135.5	 3	 14.181
135.5	 5	 14.505
135.5	10	 15.051
135.5	25	 16.119
135.5	50	 17.626
135.5	75	 19.658
135.5	85	 21.090
135.5	90	 22.261
135.5	95	 24.425
135.5	97	 26.227
136.5	 3	 14.214
136.5	 5	 14.540
136.5	10	 15.088
136.5	25	 16.163
136.5	50	 17.678
136.5	75	 19.723
136.5	85	 21.164
136.5	90	 22.341
136.5	95	 24.519
136.5	97	 26.332
137.5	 3	 14.248
137.5	 5	 14.575
137.5	10	 15.127
137.5	25	 16.207
137.5	50	 17.731
137.5	75	 19.788
137.5	85	 21.237
137.5	90	 22.422
137.5	95	 24.613
137.5	97	 26.437
138.5	 3	 14.282
138.5	 5	 14.611
138.5	10	 15.165
138.5	25	 16.251
138.5	50	 17.784
138.5	75	 19.852
138.5	85	 21.310
138.5	90	 22.502
138.5	95	 24.706
138.5	97	 26.541
139.5	 3	 14.316
139.5	 5	 14.647
139.5	10	 15.204
139.5	25	 16.296
139.5	50	 17.836
139.5	75	 19.916
139.5	85	 21.382
139.5	90	 22.581
139.5	95	 24.799
139.5	97	 26.645
140.5	 3	 14.351
140.5	 5	 14.683
140.5	10	 15.243
140.5	25	 16.341
140.5	50	 17.889
140.5	75	 19.981
140.5	85	 21.455
140.5	90	 22.661
140.5	95	 24.891
140.5	97	 26.748
141.5	 3	 14.386
141.5	 5	 14.720
141.5	10	 15.283
141.5	25	 16.386
141.5	50	 17.942
141.5	75	 20.045
141.5	85	 21.527
141.5	90	 22.740
141.5	95	 24.983
141.5	97	 26.851
142.5	 3	 14.422
142.5	 5	 14.757
142.5	10	 15.323
142.5	25	 16.431
142.5	50	 17.995
142.5	75	 20.109
142.5	85	 21.599
142.5	90	 22.819
142.5	95	 25.074
142.5	97	 26.953
143.5	 3	 14.458
143.5	 5	 14.795
143.5	10	 15.363
143.5	25	 16.476
143.5	50	 18.048
143.5	75	 20.173
143.5	85	 21.671
143.5	90	 22.897
143.5	95	 25.165
143.5	97	 27.055
144.5	 3	 14.494
144.5	 5	 14.833
144.5	10	 15.403
144.5	25	 16.522
144.5	50	 18.101
144.5	75	 20.236
144.5	85	 21.743
144.5	90	 22.975
144.5	95	 25.256
144.5	97	 27.156
145.5	 3	 14.531
145.5	 5	 14.871
145.5	10	 15.444
145.5	25	 16.568
145.5	50	 18.155
145.5	75	 20.300
145.5	85	 21.814
145.5	90	 23.053
145.5	95	 25.346
145.5	97	 27.256
146.5	 3	 14.568
146.5	 5	 14.909
146.5	10	 15.485
146.5	25	 16.613
146.5	50	 18.208
146.5	75	 20.363
146.5	85	 21.885
146.5	90	 23.130
146.5	95	 25.435
146.5	97	 27.356
147.5	 3	 14.605
147.5	 5	 14.948
147.5	10	 15.526
147.5	25	 16.659
147.5	50	 18.261
147.5	75	 20.427
147.5	85	 21.955
147.5	90	 23.207
147.5	95	 25.524
147.5	97	 27.456
148.5	 3	 14.643
148.5	 5	 14.987
148.5	10	 15.567
148.5	25	 16.705
148.5	50	 18.314
148.5	75	 20.490
148.5	85	 22.026
148.5	90	 23.283
148.5	95	 25.612
148.5	97	 27.554
149.5	 3	 14.681
149.5	 5	 15.026
149.5	10	 15.609
149.5	25	 16.752
149.5	50	 18.367
149.5	75	 20.552
149.5	85	 22.096
149.5	90	 23.359
149.5	95	 25.700
149.5	97	 27.653
150.5	 3	 14.719
150.5	 5	 15.066
150.5	10	 15.650
150.5	25	 16.798
150.5	50	 18.420
150.5	75	 20.615
150.5	85	 22.165
150.5	90	 23.435
150.5	95	 25.787
150.5	97	 27.750
151.5	 3	 14.757
151.5	 5	 15.105
151.5	10	 15.692
151.5	25	 16.844
151.5	50	 18.473
151.5	75	 20.677
151.5	85	 22.235
151.5	90	 23.510
151.5	95	 25.874
151.5	97	 27.847
152.5	 3	 14.796
152.5	 5	 15.145
152.5	10	 15.735
152.5	25	 16.891
152.5	50	 18.526
152.5	75	 20.739
152.5	85	 22.303
152.5	90	 23.585
152.5	95	 25.960
152.5	97	 27.943
153.5	 3	 14.835
153.5	 5	 15.186
153.5	10	 15.777
153.5	25	 16.937
153.5	50	 18.579
153.5	75	 20.801
153.5	85	 22.372
153.5	90	 23.659
153.5	95	 26.046
153.5	97	 28.039
154.5	 3	 14.874
154.5	 5	 15.226
154.5	10	 15.819
154.5	25	 16.984
154.5	50	 18.631
154.5	75	 20.863
154.5	85	 22.440
154.5	90	 23.733
154.5	95	 26.131
154.5	97	 28.134
155.5	 3	 14.914
155.5	 5	 15.267
155.5	10	 15.862
155.5	25	 17.031
155.5	50	 18.684
155.5	75	 20.924
155.5	85	 22.508
155.5	90	 23.806
155.5	95	 26.215
155.5	97	 28.228
156.5	 3	 14.953
156.5	 5	 15.307
156.5	10	 15.905
156.5	25	 17.077
156.5	50	 18.736
156.5	75	 20.985
156.5	85	 22.575
156.5	90	 23.879
156.5	95	 26.299
156.5	97	 28.322
157.5	 3	 14.993
157.5	 5	 15.348
157.5	10	 15.948
157.5	25	 17.124
157.5	50	 18.789
157.5	75	 21.045
157.5	85	 22.642
157.5	90	 23.951
157.5	95	 26.382
157.5	97	 28.415
158.5	 3	 15.033
158.5	 5	 15.390
158.5	10	 15.991
158.5	25	 17.171
158.5	50	 18.841
158.5	75	 21.106
158.5	85	 22.708
158.5	90	 24.023
158.5	95	 26.465
158.5	97	 28.507
159.5	 3	 15.074
159.5	 5	 15.431
159.5	10	 16.034
159.5	25	 17.217
159.5	50	 18.893
159.5	75	 21.165
159.5	85	 22.774
159.5	90	 24.094
159.5	95	 26.546
159.5	97	 28.599
160.5	 3	 15.114
160.5	 5	 15.472
160.5	10	 16.077
160.5	25	 17.264
160.5	50	 18.945
160.5	75	 21.225
160.5	85	 22.840
160.5	90	 24.165
160.5	95	 26.628
160.5	97	 28.690
161.5	 3	 15.155
161.5	 5	 15.514
161.5	10	 16.120
161.5	25	 17.311
161.5	50	 18.997
161.5	75	 21.284
161.5	85	 22.905
161.5	90	 24.235
161.5	95	 26.709
161.5	97	 28.781
162.5	 3	 15.195
162.5	 5	 15.556
162.5	10	 16.163
162.5	25	 17.357
162.5	50	 19.048
162.5	75	 21.343
162.5	85	 22.970
162.5	90	 24.305
162.5	95	 26.789
162.5	97	 28.871
163.5	 3	 15.236
163.5	 5	 15.598
163.5	10	 16.207
163.5	25	 17.404
163.5	50	 19.099
163.5	75	 21.402
163.5	85	 23.034
163.5	90	 24.374
163.5	95	 26.868
163.5	97	 28.960
164.5	 3	 15.277
164.5	 5	 15.639
164.5	10	 16.250
164.5	25	 17.450
164.5	50	 19.151
164.5	75	 21.460
164.5	85	 23.097
164.5	90	 24.443
164.5	95	 26.947
164.5	97	 29.049
165.5	 3	 15.319
165.5	 5	 15.681
165.5	10	 16.294
165.5	25	 17.497
165.5	50	 19.201
165.5	75	 21.517
165.5	85	 23.160
165.5	90	 24.511
165.5	95	 27.025
165.5	97	 29.137
166.5	 3	 15.360
166.5	 5	 15.723
166.5	10	 16.337
166.5	25	 17.543
166.5	50	 19.252
166.5	75	 21.575
166.5	85	 23.223
166.5	90	 24.578
166.5	95	 27.103
166.5	97	 29.224
167.5	 3	 15.401
167.5	 5	 15.765
167.5	10	 16.380
167.5	25	 17.589
167.5	50	 19.302
167.5	75	 21.632
167.5	85	 23.285
167.5	90	 24.645
167.5	95	 27.180
167.5	97	 29.311
168.5	 3	 15.442
168.5	 5	 15.808
168.5	10	 16.424
168.5	25	 17.635
168.5	50	 19.353
168.5	75	 21.688
168.5	85	 23.347
168.5	90	 24.711
168.5	95	 27.256
168.5	97	 29.397
169.5	 3	 15.484
169.5	 5	 15.850
169.5	10	 16.467
169.5	25	 17.681
169.5	50	 19.402
169.5	75	 21.744
169.5	85	 23.408
169.5	90	 24.777
169.5	95	 27.332
169.5	97	 29.482
170.5	 3	 15.525
170.5	 5	 15.892
170.5	10	 16.510
170.5	25	 17.727
170.5	50	 19.452
170.5	75	 21.800
170.5	85	 23.469
170.5	90	 24.842
170.5	95	 27.407
170.5	97	 29.567
171.5	 3	 15.567
171.5	 5	 15.934
171.5	10	 16.554
171.5	25	 17.772
171.5	50	 19.501
171.5	75	 21.855
171.5	85	 23.529
171.5	90	 24.907
171.5	95	 27.481
171.5	97	 29.652
172.5	 3	 15.608
172.5	 5	 15.976
172.5	10	 16.597
172.5	25	 17.818
172.5	50	 19.550
172.5	75	 21.910
172.5	85	 23.588
172.5	90	 24.971
172.5	95	 27.555
172.5	97	 29.735
173.5	 3	 15.649
173.5	 5	 16.018
173.5	10	 16.640
173.5	25	 17.863
173.5	50	 19.599
173.5	75	 21.964
173.5	85	 23.647
173.5	90	 25.034
173.5	95	 27.628
173.5	97	 29.818
174.5	 3	 15.691
174.5	 5	 16.060
174.5	10	 16.683
174.5	25	 17.908
174.5	50	 19.647
174.5	75	 22.018
174.5	85	 23.706
174.5	90	 25.097
174.5	95	 27.701
174.5	97	 29.901
175.5	 3	 15.732
175.5	 5	 16.102
175.5	10	 16.726
175.5	25	 17.953
175.5	50	 19.696
175.5	75	 22.071
175.5	85	 23.764
175.5	90	 25.160
175.5	95	 27.773
175.5	97	 29.983
176.5	 3	 15.774
176.5	 5	 16.144
176.5	10	 16.768
176.5	25	 17.998
176.5	50	 19.743
176.5	75	 22.124
176.5	85	 23.821
176.5	90	 25.221
176.5	95	 27.844
176.5	97	 30.065
177.5	 3	 15.815
177.5	 5	 16.185
177.5	10	 16.811
177.5	25	 18.042
177.5	50	 19.791
177.5	75	 22.177
177.5	85	 23.878
177.5	90	 25.282
177.5	95	 27.915
177.5	97	 30.145
178.5	 3	 15.856
178.5	 5	 16.227
178.5	10	 16.853
178.5	25	 18.086
178.5	50	 19.838
178.5	75	 22.229
178.5	85	 23.934
178.5	90	 25.343
178.5	95	 27.985
178.5	97	 30.226
179.5	 3	 15.897
179.5	 5	 16.268
179.5	10	 16.896
179.5	25	 18.130
179.5	50	 19.884
179.5	75	 22.280
179.5	85	 23.990
179.5	90	 25.403
179.5	95	 28.055
179.5	97	 30.306
180.5	 3	 15.938
180.5	 5	 16.310
180.5	10	 16.938
180.5	25	 18.174
180.5	50	 19.931
180.5	75	 22.331
180.5	85	 24.045
180.5	90	 25.462
180.5	95	 28.124
180.5	97	 30.385
181.5	 3	 15.979
181.5	 5	 16.351
181.5	10	 16.980
181.5	25	 18.217
181.5	50	 19.976
181.5	75	 22.381
181.5	85	 24.100
181.5	90	 25.521
181.5	95	 28.192
181.5	97	 30.464
182.5	 3	 16.019
182.5	 5	 16.392
182.5	10	 17.021
182.5	25	 18.260
182.5	50	 20.022
182.5	75	 22.431
182.5	85	 24.154
182.5	90	 25.579
182.5	95	 28.260
182.5	97	 30.543
183.5	 3	 16.060
183.5	 5	 16.433
183.5	10	 17.062
183.5	25	 18.303
183.5	50	 20.067
183.5	75	 22.481
183.5	85	 24.207
183.5	90	 25.637
183.5	95	 28.327
183.5	97	 30.621
184.5	 3	 16.100
184.5	 5	 16.473
184.5	10	 17.104
184.5	25	 18.345
184.5	50	 20.112
184.5	75	 22.530
184.5	85	 24.260
184.5	90	 25.694
184.5	95	 28.394
184.5	97	 30.698
185.5	 3	 16.140
185.5	 5	 16.514
185.5	10	 17.144
185.5	25	 18.387
185.5	50	 20.156
185.5	75	 22.578
185.5	85	 24.313
185.5	90	 25.750
185.5	95	 28.460
185.5	97	 30.775
186.5	 3	 16.180
186.5	 5	 16.554
186.5	10	 17.185
186.5	25	 18.429
186.5	50	 20.200
186.5	75	 22.626
186.5	85	 24.364
186.5	90	 25.806
186.5	95	 28.526
186.5	97	 30.852
187.5	 3	 16.219
187.5	 5	 16.593
187.5	10	 17.225
187.5	25	 18.470
187.5	50	 20.243
187.5	75	 22.673
187.5	85	 24.416
187.5	90	 25.861
187.5	95	 28.591
187.5	97	 30.928
188.5	 3	 16.259
188.5	 5	 16.633
188.5	10	 17.265
188.5	25	 18.511
188.5	50	 20.286
188.5	75	 22.720
188.5	85	 24.466
188.5	90	 25.916
188.5	95	 28.656
188.5	97	 31.004
189.5	 3	 16.298
189.5	 5	 16.672
189.5	10	 17.305
189.5	25	 18.552
189.5	50	 20.329
189.5	75	 22.766
189.5	85	 24.517
189.5	90	 25.970
189.5	95	 28.720
189.5	97	 31.080
190.5	 3	 16.336
190.5	 5	 16.711
190.5	10	 17.344
190.5	25	 18.592
190.5	50	 20.371
190.5	75	 22.812
190.5	85	 24.566
190.5	90	 26.024
190.5	95	 28.784
190.5	97	 31.155
191.5	 3	 16.375
191.5	 5	 16.750
191.5	10	 17.383
191.5	25	 18.632
191.5	50	 20.412
191.5	75	 22.857
191.5	85	 24.615
191.5	90	 26.077
191.5	95	 28.847
191.5	97	 31.230
192.5	 3	 16.413
192.5	 5	 16.788
192.5	10	 17.422
192.5	25	 18.671
192.5	50	 20.453
192.5	75	 22.902
192.5	85	 24.664
192.5	90	 26.129
192.5	95	 28.910
192.5	97	 31.305
193.5	 3	 16.450
193.5	 5	 16.826
193.5	10	 17.460
193.5	25	 18.710
193.5	50	 20.494
193.5	75	 22.946
193.5	85	 24.712
193.5	90	 26.181
193.5	95	 28.972
193.5	97	 31.379
194.5	 3	 16.488
194.5	 5	 16.863
194.5	10	 17.498
194.5	25	 18.749
194.5	50	 20.534
194.5	75	 22.990
194.5	85	 24.759
194.5	90	 26.233
194.5	95	 29.034
194.5	97	 31.453
195.5	 3	 16.525
195.5	 5	 16.900
195.5	10	 17.535
195.5	25	 18.787
195.5	50	 20.573
195.5	75	 23.033
195.5	85	 24.806
195.5	90	 26.284
195.5	95	 29.096
195.5	97	 31.527
196.5	 3	 16.561
196.5	 5	 16.937
196.5	10	 17.572
196.5	25	 18.824
196.5	50	 20.613
196.5	75	 23.076
196.5	85	 24.852
196.5	90	 26.334
196.5	95	 29.157
196.5	97	 31.601
197.5	 3	 16.597
197.5	 5	 16.973
197.5	10	 17.608
197.5	25	 18.861
197.5	50	 20.651
197.5	75	 23.118
197.5	85	 24.898
197.5	90	 26.384
197.5	95	 29.217
197.5	97	 31.674
198.5	 3	 16.633
198.5	 5	 17.009
198.5	10	 17.644
198.5	25	 18.898
198.5	50	 20.689
198.5	75	 23.159
198.5	85	 24.944
198.5	90	 26.434
198.5	95	 29.278
198.5	97	 31.748
199.5	 3	 16.668
199.5	 5	 17.044
199.5	10	 17.680
199.5	25	 18.934
199.5	50	 20.727
199.5	75	 23.200
199.5	85	 24.988
199.5	90	 26.483
199.5	95	 29.338
199.5	97	 31.821
200.5	 3	 16.703
200.5	 5	 17.079
200.5	10	 17.715
200.5	25	 18.969
200.5	50	 20.764
200.5	75	 23.241
200.5	85	 25.033
200.5	90	 26.532
200.5	95	 29.398
200.5	97	 31.894
201.5	 3	 16.737
201.5	 5	 17.113
201.5	10	 17.749
201.5	25	 19.004
201.5	50	 20.800
201.5	75	 23.281
201.5	85	 25.076
201.5	90	 26.580
201.5	95	 29.457
201.5	97	 31.967
202.5	 3	 16.771
202.5	 5	 17.147
202.5	10	 17.783
202.5	25	 19.039
202.5	50	 20.836
202.5	75	 23.320
202.5	85	 25.120
202.5	90	 26.627
202.5	95	 29.516
202.5	97	 32.040
203.5	 3	 16.804
203.5	 5	 17.180
203.5	10	 17.816
203.5	25	 19.073
203.5	50	 20.871
203.5	75	 23.359
203.5	85	 25.163
203.5	90	 26.674
203.5	95	 29.575
203.5	97	 32.113
204.5	 3	 16.836
204.5	 5	 17.212
204.5	10	 17.849
204.5	25	 19.106
204.5	50	 20.906
204.5	75	 23.397
204.5	85	 25.205
204.5	90	 26.721
204.5	95	 29.634
204.5	97	 32.186
205.5	 3	 16.868
205.5	 5	 17.244
205.5	10	 17.881
205.5	25	 19.139
205.5	50	 20.940
205.5	75	 23.435
205.5	85	 25.247
205.5	90	 26.768
205.5	95	 29.692
205.5	97	 32.259
206.5	 3	 16.899
206.5	 5	 17.276
206.5	10	 17.912
206.5	25	 19.171
206.5	50	 20.973
206.5	75	 23.472
206.5	85	 25.288
206.5	90	 26.814
206.5	95	 29.750
206.5	97	 32.332
207.5	 3	 16.930
207.5	 5	 17.306
207.5	10	 17.943
207.5	25	 19.202
207.5	50	 21.006
207.5	75	 23.509
207.5	85	 25.329
207.5	90	 26.859
207.5	95	 29.808
207.5	97	 32.405
208.5	 3	 16.960
208.5	 5	 17.336
208.5	10	 17.973
208.5	25	 19.233
208.5	50	 21.039
208.5	75	 23.545
208.5	85	 25.369
208.5	90	 26.904
208.5	95	 29.866
208.5	97	 32.479
209.5	 3	 16.989
209.5	 5	 17.366
209.5	10	 18.003
209.5	25	 19.263
209.5	50	 21.070
209.5	75	 23.580
209.5	85	 25.409
209.5	90	 26.949
209.5	95	 29.924
209.5	97	 32.552
210.5	 3	 17.017
210.5	 5	 17.394
210.5	10	 18.032
210.5	25	 19.293
210.5	50	 21.102
210.5	75	 23.616
210.5	85	 25.449
210.5	90	 26.994
210.5	95	 29.981
210.5	97	 32.626
211.5	 3	 17.045
211.5	 5	 17.422
211.5	10	 18.060
211.5	25	 19.321
211.5	50	 21.132
211.5	75	 23.650
211.5	85	 25.488
211.5	90	 27.038
211.5	95	 30.039
211.5	97	 32.700
212.5	 3	 17.072
212.5	 5	 17.449
212.5	10	 18.087
212.5	25	 19.349
212.5	50	 21.162
212.5	75	 23.684
212.5	85	 25.527
212.5	90	 27.082
212.5	95	 30.096
212.5	97	 32.774
213.5	 3	 17.099
213.5	 5	 17.476
213.5	10	 18.114
213.5	25	 19.377
213.5	50	 21.191
213.5	75	 23.718
213.5	85	 25.565
213.5	90	 27.125
213.5	95	 30.153
213.5	97	 32.848
214.5	 3	 17.124
214.5	 5	 17.501
214.5	10	 18.140
214.5	25	 19.404
214.5	50	 21.220
214.5	75	 23.751
214.5	85	 25.603
214.5	90	 27.169
214.5	95	 30.211
214.5	97	 32.923
215.5	 3	 17.149
215.5	 5	 17.526
215.5	10	 18.165
215.5	25	 19.430
215.5	50	 21.248
215.5	75	 23.784
215.5	85	 25.641
215.5	90	 27.212
215.5	95	 30.268
215.5	97	 32.998
216.5	 3	 17.173
216.5	 5	 17.550
216.5	10	 18.189
216.5	25	 19.455
216.5	50	 21.275
216.5	75	 23.816
216.5	85	 25.678
216.5	90	 27.254
216.5	95	 30.326
216.5	97	 33.073
217.5	 3	 17.196
217.5	 5	 17.573
217.5	10	 18.213
217.5	25	 19.479
217.5	50	 21.302
217.5	75	 23.847
217.5	85	 25.715
217.5	90	 27.297
217.5	95	 30.383
217.5	97	 33.149
218.5	 3	 17.218
218.5	 5	 17.596
218.5	10	 18.236
218.5	25	 19.503
218.5	50	 21.328
218.5	75	 23.878
218.5	85	 25.751
218.5	90	 27.339
218.5	95	 30.441
218.5	97	 33.225
219.5	 3	 17.239
219.5	 5	 17.617
219.5	10	 18.257
219.5	25	 19.526
219.5	50	 21.353
219.5	75	 23.909
219.5	85	 25.787
219.5	90	 27.381
219.5	95	 30.498
219.5	97	 33.302
220.5	 3	 17.259
220.5	 5	 17.637
220.5	10	 18.278
220.5	25	 19.548
220.5	50	 21.378
220.5	75	 23.939
220.5	85	 25.823
220.5	90	 27.423
220.5	95	 30.556
220.5	97	 33.380
221.5	 3	 17.278
221.5	 5	 17.657
221.5	10	 18.298
221.5	25	 19.570
221.5	50	 21.402
221.5	75	 23.969
221.5	85	 25.859
221.5	90	 27.465
221.5	95	 30.614
221.5	97	 33.458
222.5	 3	 17.296
222.5	 5	 17.675
222.5	10	 18.317
222.5	25	 19.590
222.5	50	 21.425
222.5	75	 23.998
222.5	85	 25.894
222.5	90	 27.507
222.5	95	 30.673
222.5	97	 33.536
223.5	 3	 17.314
223.5	 5	 17.693
223.5	10	 18.336
223.5	25	 19.610
223.5	50	 21.448
223.5	75	 24.027
223.5	85	 25.929
223.5	90	 27.548
223.5	95	 30.731
223.5	97	 33.615
224.5	 3	 17.330
224.5	 5	 17.709
224.5	10	 18.353
224.5	25	 19.629
224.5	50	 21.470
224.5	75	 24.055
224.5	85	 25.964
224.5	90	 27.590
224.5	95	 30.790
224.5	97	 33.695
225.5	 3	 17.345
225.5	 5	 17.725
225.5	10	 18.369
225.5	25	 19.647
225.5	50	 21.491
225.5	75	 24.083
225.5	85	 25.998
225.5	90	 27.631
225.5	95	 30.849
225.5	97	 33.776
226.5	 3	 17.359
226.5	 5	 17.739
226.5	10	 18.384
226.5	25	 19.664
226.5	50	 21.512
226.5	75	 24.110
226.5	85	 26.032
226.5	90	 27.672
226.5	95	 30.909
226.5	97	 33.858
227.5	 3	 17.372
227.5	 5	 17.753
227.5	10	 18.398
227.5	25	 19.680
227.5	50	 21.532
227.5	75	 24.137
227.5	85	 26.066
227.5	90	 27.714
227.5	95	 30.968
227.5	97	 33.940
228.5	 3	 17.384
228.5	 5	 17.765
228.5	10	 18.412
228.5	25	 19.695
228.5	50	 21.551
228.5	75	 24.164
228.5	85	 26.100
228.5	90	 27.755
228.5	95	 31.029
228.5	97	 34.023
229.5	 3	 17.395
229.5	 5	 17.776
229.5	10	 18.424
229.5	25	 19.709
229.5	50	 21.569
229.5	75	 24.190
229.5	85	 26.134
229.5	90	 27.796
229.5	95	 31.090
229.5	97	 34.107
230.5	 3	 17.404
230.5	 5	 17.786
230.5	10	 18.435
230.5	25	 19.723
230.5	50	 21.587
230.5	75	 24.216
230.5	85	 26.167
230.5	90	 27.838
230.5	95	 31.151
230.5	97	 34.192
231.5	 3	 17.412
231.5	 5	 17.795
231.5	10	 18.445
231.5	25	 19.735
231.5	50	 21.604
231.5	75	 24.241
231.5	85	 26.200
231.5	90	 27.879
231.5	95	 31.213
231.5	97	 34.278
232.5	 3	 17.419
232.5	 5	 17.803
232.5	10	 18.454
232.5	25	 19.747
232.5	50	 21.620
232.5	75	 24.266
232.5	85	 26.233
232.5	90	 27.921
232.5	95	 31.275
232.5	97	 34.365
233.5	 3	 17.425
233.5	 5	 17.809
233.5	10	 18.461
233.5	25	 19.757
233.5	50	 21.635
233.5	75	 24.290
233.5	85	 26.266
233.5	90	 27.962
233.5	95	 31.338
233.5	97	 34.453
234.5	 3	 17.430
234.5	 5	 17.815
234.5	10	 18.468
234.5	25	 19.767
234.5	50	 21.650
234.5	75	 24.314
234.5	85	 26.299
234.5	90	 28.004
234.5	95	 31.402
234.5	97	 34.543
235.5	 3	 17.433
235.5	 5	 17.819
235.5	10	 18.473
235.5	25	 19.775
235.5	50	 21.664
235.5	75	 24.338
235.5	85	 26.332
235.5	90	 28.046
235.5	95	 31.466
235.5	97	 34.633
236.5	 3	 17.435
236.5	 5	 17.821
236.5	10	 18.477
236.5	25	 19.782
236.5	50	 21.677
236.5	75	 24.361
236.5	85	 26.365
236.5	90	 28.088
236.5	95	 31.531
236.5	97	 34.725
237.5	 3	 17.435
237.5	 5	 17.823
237.5	10	 18.480
237.5	25	 19.789
237.5	50	 21.689
237.5	75	 24.384
237.5	85	 26.397
237.5	90	 28.130
237.5	95	 31.597
237.5	97	 34.818
238.5	 3	 17.434
238.5	 5	 17.823
238.5	10	 18.482
238.5	25	 19.794
238.5	50	 21.701
238.5	75	 24.407
238.5	85	 26.430
238.5	90	 28.173
238.5	95	 31.663
238.5	97	 34.913
239.5	 3	 17.432
239.5	 5	 17.821
239.5	10	 18.482
239.5	25	 19.798
239.5	50	 21.712
239.5	75	 24.429
239.5	85	 26.462
239.5	90	 28.216
239.5	95	 31.731
239.5	97	 35.008
240.0	 3	 17.430
240.0	 5	 17.820
240.0	10	 18.482
240.0	25	 19.800
240.0	50	 21.717
240.0	75	 24.440
240.0	85	 26.479
240.0	90	 28.237
240.0	95	 31.765
240.0	97	 35.057
240.5	 3	 17.428
240.5	 5	 17.819
240.5	10	 18.481
240.5	25	 19.802
240.5	50	 21.722
240.5	75	 24.451
240.5	85	 26.495
240.5	90	 28.259
240.5	95	 31.799
240.5	97	 35.106
