package Sisimai::Reason::VirusDetected;
use v5.26;
use strict;
use warnings;

sub text  { 'virusdetected' }
sub description { 'Email rejected due to a virus scanner on a destination host' }
sub match {
    # Try to match that the given text and regular expressions
    # @param    [String] argv1  String to be matched with regular expressions
    # @return   [Integer]       0: Did not match
    #                           1: Matched
    # @since v4.22.0
    my $class = shift;
    my $argv1 = shift // return undef;

    state $index = [
        'it has a potentially executable attachment',
        'the message was rejected because it contains prohibited virus or spam content',
        'this form of attachment has been used by recent viruses or other malware',
        'virus detected',
        'virus phishing/malicious_url detected',
        'your message was infected with a virus',
    ];
    return 1 if grep { rindex($argv1, $_) > -1 } @$index;
    return 0;
}

sub true {
    # The bounce reason is "virusdetected" or not
    # @param    [Sisimai::Fact] argvs   Object to be detected the reason
    # @return   [Integer]               1: virus detected
    #                                   0: virus was not detected
    # @since v4.22.0
    # @see http://www.ietf.org/rfc/rfc2822.txt
    my $class = shift;
    my $argvs = shift // return undef;

    # The value of "reason" isn't "virusdetected" when the value of "smtpcommand" is an SMTP com-
    # mand to be sent before the SMTP DATA command because all the MTAs read the headers and the
    # entire message body after the DATA command.
    return 1 if $argvs->{'reason'} eq 'virusdetected';
    return 0 if $argvs->{'smtpcommand'} eq 'CONN' || $argvs->{'smtpcommand'} eq 'HELO'
             || $argvs->{'smtpcommand'} eq 'HELO' || $argvs->{'smtpcommand'} eq 'MAIL'
             || $argvs->{'smtpcommand'} eq 'RCPT';
    return 1 if __PACKAGE__->match(lc $argvs->{'diagnosticcode'});
    return 0;
}

1;
__END__

=encoding utf-8

=head1 NAME

Sisimai::Reason::VirusDetected - Bounce reason is C<virusdetected> or not.

=head1 SYNOPSIS

    use Sisimai::Reason::VirusDetected;
    print Sisimai::Reason::VirusDetected->match('5.7.1 Email not accept');   # 1

=head1 DESCRIPTION

Sisimai::Reason::VirusDetected checks the bounce reason is C<virusdetected> or not. This class is
called only Sisimai::Reason class.

This is an error that any virus or trojan horse detected in the message by a virus scanner program
at a destination mail server. This reason has been divided from C<securityerror> at Sisimai 4.22.0.

    Your message was infected with a virus. You should download a virus
    scanner and check your computer for viruses.

    Sender:    <sironeko@libsisimai.org>
    Recipient: <kijitora@example.jp>

=head1 CLASS METHODS

=head2 C<B<text()>>

C<text()> returns string: C<virusdetected>.

    print Sisimai::Reason::VirusDetected->text;  # virusdetected

=head2 C<B<match(I<string>)>>

C<match()> returns 1 if the argument matched with patterns defined in this class.

    my $v = 'Your message was infected with a virus. ...';
    print Sisimai::Reason::VirusDetected->match($v);    # 1

=head2 C<B<true(I<Sisimai::Fact>)>>

C<true()> returns 1 if the bounce reason is C<virusdetected>. The argument must be Sisimai::Fact
object and this method is called only from Sisimai::Reason class.

=head1 AUTHOR

azumakuniyuki

=head1 COPYRIGHT

Copyright (C) 2017-2021,2023,2024 azumakuniyuki, All rights reserved.

=head1 LICENSE

This software is distributed under The BSD 2-Clause License.

=cut
